<?php
/**
 * @package       RSMediaGallery! Responsive Slider
 * @copyright (C) 2014 www.rsjoomla.com
 * @license       GPL, http://www.gnu.org/copyleft/gpl.html
 */

defined('_JEXEC') or die('Restricted access');
require_once JPATH_ADMINISTRATOR . '/components/com_rsmediagallery/helpers/helper.php';
require_once JPATH_ADMINISTRATOR . '/components/com_rsmediagallery/helpers/jquery.php';

/**
 * Class RSMGResponsiveSlider
 */
class RSMGResponsiveSlider
{
	/**
	 * @var
	 */
	private $type;
	
	/**
	 * @var array
	 */
	private $options;
	
	/**
	 * @var array|instagramSlider|mixed
	 */
	private $items;
	
	/**
	 * @var
	 */
	private $selector;

	/**
	 * @var
	 */
	private $params;
	
	/**
	 * @var
	 */
	private $position;
	
	/**
	 * RSMGResponsiveSlider constructor.
	 *
	 * @param $params
	 * @param $selector
	 */
	public function __construct($params, $selector, $position)
	{
		$this->params   = $params;
		$this->selector = $selector;
		$this->type     = $params->get('sliderType', 'rsmg');

		$this->items    = $this->getSliderItems($this->type, $params);
		$this->options  = $this->buildOptions($params, $this->type, $this->items);
		$this->position = $position;
	}

	/**
	 * When changing the type of slider,
	 * we need to re-create the object
	 *
	 * @param $type
	 *
	 * @throws Exception
	 */
	public function setType($type)
	{
		$this->type    = $type;
		$this->items   = $this->getSliderItems($this->type, $this->params);
		$this->options = $this->buildOptions($this->params, $this->type, $this->items);
	}

	/**
	 * @return mixed
	 */
	public function getType()
	{
		return $this->type;
	}

	/**
	 * @return array
	 */
	public function getOptions()
	{
		return $this->options;
	}

	/**
	 * @param array $options
	 */
	public function setOptions($options)
	{
		$this->options = $options;
	}

	/**
	 * @return array|instagramSlider|mixed
	 */
	public function getItems()
	{
		return $this->items;
	}

	/**
	 * @param array|instagramSlider|mixed $items
	 */
	public function setItems($items)
	{
		$this->items = $items;
	}

	/**
	 * @return mixed
	 */
	public function getSelector()
	{
		return $this->selector;
	}

	/**
	 * @param mixed $selector
	 */
	public function setSelector($selector)
	{
		$this->selector = $selector;
	}

	/**
	 * @param $type
	 * @param $params
	 *
	 * @return array|instagramSlider|mixed
	 * @throws Exception
	 */
	private function getSliderItems($type, $params)
	{
		$items = array();
		switch ($type)
		{
			case 'rsmg':

				if (!$params->get('tags'))
				{
					throw new Exception(JText::sprintf('MOD_RSMEDIAGALLERY_RESPONSIVE_SLIDER_PARAM_TAGS_NO_RESULTS', $params->get('tags')));
				}

				$items = RSMediaGalleryHelper::getItems($params->get('tags'), $params->get('ordering', 'ordering'), $params->get('direction', 'ASC'), 0, (int) $params->get('limit', 5));

				if (!count($items))
				{
					throw new Exception(JText::sprintf('MOD_RSMEDIAGALLERY_RESPONSIVE_SLIDER_PARAM_TAGS_NO_RESULTS', $params->get('tags')));
				}

				break;
			case 'instagram':

				require_once dirname(__FILE__) . '/instagram.php';
				$items = new instagramSlider($params);
				$items = $items->getItems();

				if (!count($items))
				{
					throw new Exception(JText::_('MOD_RSMEDIAGALLERY_RESPONSIVE_SLIDER_INSTAGRAM_NO_RESULTS'));
				}

				break;
			case 'articles':
				require_once JPATH_SITE . '/components/com_content/helpers/route.php';

				$db    = JFactory::getDbo();
				$query = $db->getQuery(true);

				$query->select('*')
					->from($db->qn('#__content'))
					->where($db->qn('featured') . ' = ' . $db->q(1));

				$db->setQuery($query);
				$result = $db->loadObjectList('id');

				foreach ($result as $article)
				{
					$image = json_decode($article->images);

					$items[] = array_filter(array(
						'id'    => $article->id,
						'title' => $article->title,
						'intro' => strip_tags($article->introtext),
						'image' => $image->image_fulltext,
						'url'   => ContentHelperRoute::getArticleRoute($article->id),
					));

				}

				foreach ($items as $item => $properties)
				{
					if (!isset($properties['image']))
					{
						unset($items[$item]);
					}
				}

				if (!count($items))
				{
					throw new Exception(JText::_('MOD_RSMEDIAGALLERY_RESPONSIVE_SLIDER_NO_ARTICLES_RESULTS'));
				}

				break;
		}

		if ((int) $params->get('limit') != 0 && count($items) > (int) $params->get('limit'))
		{
			$items = array_slice($items, 0, (int) $params->get('limit'));
		}

		return $items;
	}

	/**
	 * @param $params
	 * @param $type
	 * @param $items
	 *
	 * @return array
	 */
	private function buildOptions($params, $type, $items)
	{
		/**
		 * The array holding the options
		 */
		$options = array(
			'autoplay'      => (bool) $params->get('autoplay', false),
			'infinite'      => (bool) $params->get('loop', true),
			'pauseOnHover'  => (bool) $params->get('stop_over', true),
			'autoplaySpeed' => (int) $params->get('interval', 5000),
			'arrows'        => (bool) $params->get('show_controls', true),
			'dots'          => (bool) $params->get('show_indicators', true),
			'centerMode'    => (bool) false,
			'fade'          => (bool) $params->get('transition_in'),

			// Defaults
			'draggable'     => false,
			'prevArrow'     => '"<div class=\"rstpl-car-previous\"></div>"',
			'nextArrow'     => '"<div class=\"rstpl-car-next\"></div>"',
			'responsive'    => '[{breakpoint:1,settings:{arrows:false}},{breakpoint:980,settings: {arrows:false}}]',
		);

		if ($options['arrows'])
		{
			$options['responsive'] = '[{breakpoint:768,settings:{arrows:false}},{breakpoint:980,settings: {arrows:true}}]';
		}

		/**
		 * In case we only have 1 item in the slider,
		 * we need to remove autoplay, controls and dots
		 */
		if (count($items) == 1 || $params->get('limit') == 1)
		{
			$options['autoplay'] = false;
			$options['infinite'] = false;
			$options['arrows']   = false;
			$options['dots']     = false;
		}

		/**
		 * Instagram slider needs particular settings
		 */
		if ($type == 'instagram')
		{
			$options['centerMode']    = true;
			$options['slideToShow']   = '3';
			$options['centerPadding'] = '"25%"';
			if ($options['arrows'])
			{
				$options['responsive']    = '[{breakpoint:480,settings:{arrows:true, slidesToShow:1}},{breakpoint:980,settings: {arrows:true}}]';
			}	
			$options['fade']          = false;
		}

		return $options;
	}

	/**
	 * @param $type
	 * @param $items
	 * @param $params
	 */
	public function launchSlider($type, $items, $params)
	{
		$document = JFactory::getDocument();
		$this->loadFrontendLibraries();

		/**
		 * Frontend style, functionality settings
		 */
		$enableParallax      = (bool) $params->get('parallaxEffects', '1');
		$show_title          = (int) $params->get('show_title', 1);
		$show_description    = (int) $params->get('show_description', 1);
		$show_url            = (int) $params->get('show_url', 1);
		$open_in_new_page    = (int) $params->get('open_in_new_page', 1);
		$animate_title       = $params->get('animate_title');
		$animate_description = $params->get('animate_description');
		$animate_button      = $params->get('animate_button');
		$title_color         = $params->get('title_color');
		$description_color   = $params->get('description_color');
		$button_color        = $params->get('button_color');
		$moduleclass_sfx     = $params->get('moduleclass_sfx', '');

		switch ($type)
		{
			case 'rsmg':
				$gallery_params = RSMediaGalleryHelper::parseParams($params);
				$gallery_params->set('use_original', 1);

				if ($items)
				{
					$height = '';
					foreach ($items as $i => $item)
					{
						$items[$i] = RSMediaGalleryHelper::parseItem($item, $gallery_params);
						$height    = unserialize($item->params);
						$height    = $height['info'][1];
					}

					if ($enableParallax)
					{
						$height = $height - ($height * (25 / 100));
					}

					$document->addStyleDeclaration("
						@media (min-width: 1180px) {
							.rstpl-slider-main-image{
								height:" . $height . "px;
							}
						}
					"
					);

					$caption_class = $this->getCaptionClass($params->get('status_captions', 'show-all'));
					$caption_class .= ' ' . $params->get('align_caption', 'text-center');
					$caption_class .= ' ' . $params->get('vertical_position', 'vertical-middle');
					$caption_class .= ' ' . $params->get('caption_style');

					$slider_class = $params->get('indicators_position');

					$slider_class = $params->get('indicators_position');
					if ((bool) $params->get('parallaxEffects', '1'))
					{
						$slider_class .= ' parallax-enabled ';
					}
					// Display template
					require JModuleHelper::getLayoutPath('mod_rsmediagallery_responsive_slider', $params->get('layout', 'default'));
				}
				break;

			case 'articles':
				$document = JFactory::getDocument();
				jimport('joomla.image.image');
				$image  = new Jimage($items[0]['image']);
				$height = $image->getHeight();

				if ($enableParallax)
				{
					$height = $image->getHeight() - ($image->getHeight() * (25 / 100));
				}

				$document->addStyleDeclaration("
					@media (min-width: 1180px) {
						.rstpl-slider-main-image{
							height:" . $height . "px;
						}
					}
					"
				);

				$caption_class = $this->getCaptionClass($params->get('status_captions', 'show-all'));
				$caption_class .= ' ' . $params->get('align_caption', 'text-center');
				$caption_class .= ' ' . $params->get('vertical_position', 'vertical-middle');
				$caption_class .= ' ' . $params->get('caption_style');

				$slider_class = $params->get('indicators_position');
				if ((bool) $params->get('parallaxEffects', '1'))
				{
					$slider_class .= ' parallax-enabled ';
				}

				require JModuleHelper::getLayoutPath('mod_rsmediagallery_responsive_slider', $params->get('layout', 'post-slider'));
				break;

			case 'instagram':
				$slider_class = $params->get('indicators_position') . ' instagram-slider ';
				require JModuleHelper::getLayoutPath('mod_rsmediagallery_responsive_slider', $params->get('layout', 'instagram'));

				break;
		}

		$this->setGoogleFont($params->get('googleFont'));
		$this->setOverlay($params, $this->selector);
		$document->addScriptDeclaration($this->buildJsString($this->options, $this->selector));

	}

	/**
	 * @param $font
	 *
	 * @return bool
	 */
	private function setGoogleFont($font)
	{
		if ($font === 'Template Default')
		{
			return false;
		}

		$document        = JFactory::getDocument();
		$uri             = JURI::getInstance();
		$prefix_protocol = 'http';

		if ($uri->isSSL())
		{
			$prefix_protocol .= 's';
		}

		$document->addCustomTag('<link href="' . $prefix_protocol . '://fonts.googleapis.com/css?family=' . urlencode($font) . ':300,400,700,400italic,700italic" rel="stylesheet" type="text/css" />');

		$document->addStyleDeclaration("
				.rstpl-caption p,
				.rstpl-caption h2,
				.rstpl-caption a{
					font-family:" . $font . ";
				}
			");

		return true;
	}

	/**
	 * Load the needed frontend libraries
	 */
	private function loadFrontendLibraries()
	{
		$document     = JFactory::getDocument();
		$jqueryHelper = RSMediaGalleryjQuery::getInstance();

		$jqueryHelper->addjQuery();
		$document->addScriptDeclaration("jQuery.noConflict();");

		JHtml::stylesheet('mod_rsmediagallery_responsive_slider/style.css', array(), true);
		JHtml::stylesheet('mod_rsmediagallery_responsive_slider/slick.css', array(), true);
		JHtml::script('mod_rsmediagallery_responsive_slider/slick.min.js', false, true);
		JHtml::script('mod_rsmediagallery_responsive_slider/script.js', false, true);
		JHtml::script('mod_rsmediagallery_responsive_slider/skrollr.js', false, true);
	}

	/**
	 * @param $params
	 * @param $selector
	 *
	 * @return string
	 */
	private function buildJsString($params, $selector)
	{
		$options = array();

		foreach ($params as $option => $value)
		{
			if (is_bool($value))
			{
				$options[] = $value ? $option . ':true' : $option . ':false';
			}
			else
			{
				$options[] = $option . ':' . $value . '';
			}
		}

		if (!empty($options))
		{
			$options = '{' . "\n" . implode(",\n", $options) . "\n" . '}';
		}

		return "jQuery(document).ready(function($){
					$('.$selector').slick($options);
				});";
	}

	/**
	 * @param $status
	 *
	 * @return string
	 */
	public function getCaptionClass($status)
	{
		$classes = '';

		if ($status != 'show-all')
		{
			$devices = explode('-', $status);

			if (in_array('phones', $devices))
			{
				$classes .= 'visible-phone ';
			}

			if (in_array('tablets', $devices))
			{
				$classes .= 'visible-tablet ';
			}

			if (in_array('desktop', $devices))
			{
				$classes .= 'visible-desktop ';
			}

		}

		return trim($classes);
	}

	/**
	 * @param $params
	 * @param $context
	 *
	 * @return bool
	 */
	private function setOverlay($params, $context)
	{
		$document = JFactory::getDocument();
		$type     = (int) $params->get('gradient_solid');

		if ($params->get('sliderType') == 'instagram')
		{
			$type = 2;
		}

		$args = array(
			0 => false,
			1 => array(
				'gradient_start'         => $params->get('gradient_start_color', false),
				'gradient_end'           => $params->get('gradient_end_color', false),
				'gradient_opacity_start' => $params->get('opacity_start', false),
				'gradient_opacity_end'   => $params->get('opacity_end', false),
				'gradient_direction'     => (int) $params->get('gradient_direction', 90)
			),
			2 => array(
				'color_overlay' => $params->get('overlay_color', false),
				'color_opacity' => $params->get('opacity', false),
			)
		);

		switch ($type)
		{
			case 0:
				return false;
				break;
			case 1:
				$gradient_start = $this->hex2rgba($args[$type]['gradient_start'], $args[$type]['gradient_opacity_start']);
				$gradient_end   = $this->hex2rgba($args[$type]['gradient_end'], $args[$type]['gradient_opacity_end']);
				// Create the gradient based on the options provided
				$document->addStyleDeclaration("
				." . $context . "{
					background:" . $gradient_start . ";
				}
				." . $context . " .item:before{
					background: -webkit-linear-gradient(" . $args[$type]['gradient_direction'] . "deg, " . $gradient_start . " 10%, " . $gradient_end . " 90%);
					background:    -moz-linear-gradient(" . $args[$type]['gradient_direction'] . "deg, " . $gradient_start . " 10%, " . $gradient_end . " 90%);
					background:     -ms-linear-gradient(" . $args[$type]['gradient_direction'] . "deg, " . $gradient_start . " 10%, " . $gradient_end . " 90%);
					background:      -o-linear-gradient(" . $args[$type]['gradient_direction'] . "deg, " . $gradient_start . " 10%, " . $gradient_end . " 90%);
					background:         linear-gradient(" . $args[$type]['gradient_direction'] . "deg, " . $gradient_start . " 10%, " . $gradient_end . " 90%);
				}
				");
				break;
			case 2:
				$color_overlay = $this->hex2rgba($args[$type]['color_overlay'], $args[$type]['color_opacity']);
				// Add the overlay
				$document->addStyleDeclaration("
				." . $context . "{
					background:" . $color_overlay . ";
				}
				." . $context . " .item:before{
					background-color:" . $color_overlay . ";
				}
				");
				break;
		}

		return true;
	}

	/**
	 * @param      $color
	 * @param bool $opacity
	 *
	 * @return string
	 */
	public function hex2rgba($color, $opacity = false)
	{
		$default = '#000';

		if (empty($color))
		{
			$color = $default;
		}

		if ($color[0] == '#')
		{
			$color = substr($color, 1);
		}

		if (strlen($color) == 6)
		{
			$hex = array($color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5]);
		}

		elseif (strlen($color) == 3)
		{
			$hex = array($color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2]);
		}
		else
		{
			return $default;
		}

		$rgb = array_map('hexdec', $hex);
		if ($opacity)
		{

			if (abs($opacity) > 1)
			{
				$opacity = 1.0;
			}
			$output = 'rgba(' . implode(",", $rgb) . ',' . $opacity . ')';
		}
		else
		{
			$output = 'rgb(' . implode(",", $rgb) . ')';
		}

		return $output;
	}

	public static function add_attributes(array $array)
	{
		$attrbs = array();
		foreach ($array as $property => $value)
		{

			if (strlen($value))
			{
				$attrbs[] = $property . '="' . RSMediaGalleryHelper::escape($value) . '"';
			}

		}

		if ($attrbs)
		{
			return implode(' ', $attrbs);
		}

		return false;
	}

}